<?php
/**
 * Get user's chat list
 * Returns: Array of chats with other user info and last message
 */

header('Content-Type: application/json');
require_once '../auth.php';
require_once '../db.php';

requireAuth();

$userId = (int) getCurrentUserId();
$db = getDB();

$chats = [];
try {
    // Exclude blocked users (requires blocks table)
    $stmt = $db->prepare("
        SELECT 
            c.id as chat_id,
            CASE WHEN c.user1_id = ? THEN c.user2_id ELSE c.user1_id END as other_user_id,
            u.username as other_username,
            u.profile_pic_url as other_profile_pic,
            m.message_text as last_message,
            m.media_type as last_message_type,
            m.created_at as last_message_time,
            m.id as last_message_id
        FROM chats c
        INNER JOIN users u ON (CASE WHEN c.user1_id = ? THEN u.id = c.user2_id ELSE u.id = c.user1_id END)
        LEFT JOIN blocks b ON b.blocker_id = ? AND b.blocked_id = (CASE WHEN c.user1_id = ? THEN c.user2_id ELSE c.user1_id END)
        LEFT JOIN messages m ON m.chat_id = c.id AND m.id = (SELECT id FROM messages WHERE chat_id = c.id ORDER BY created_at DESC LIMIT 1)
        WHERE (c.user1_id = ? OR c.user2_id = ?) AND b.id IS NULL
        ORDER BY m.created_at DESC, c.created_at DESC
    ");
    $stmt->execute([$userId, $userId, $userId, $userId, $userId, $userId]);
    $chats = $stmt->fetchAll();
} catch (PDOException $e) {
    // Fallback if blocks table missing
    $stmt = $db->prepare("
        SELECT 
            c.id as chat_id,
            CASE WHEN c.user1_id = ? THEN c.user2_id ELSE c.user1_id END as other_user_id,
            u.username as other_username,
            u.profile_pic_url as other_profile_pic,
            m.message_text as last_message,
            m.media_type as last_message_type,
            m.created_at as last_message_time,
            m.id as last_message_id
        FROM chats c
        INNER JOIN users u ON (CASE WHEN c.user1_id = ? THEN u.id = c.user2_id ELSE u.id = c.user1_id END)
        LEFT JOIN messages m ON m.chat_id = c.id AND m.id = (SELECT id FROM messages WHERE chat_id = c.id ORDER BY created_at DESC LIMIT 1)
        WHERE c.user1_id = ? OR c.user2_id = ?
        ORDER BY m.created_at DESC, c.created_at DESC
    ");
    $stmt->execute([$userId, $userId, $userId, $userId]);
    $chats = $stmt->fetchAll();
}

// Format response
$formattedChats = array_map(function($chat) {
    return [
        'chat_id' => (int)$chat['chat_id'],
        'other_user' => [
            'id' => (int)$chat['other_user_id'],
            'username' => $chat['other_username'],
            'profile_pic_url' => $chat['other_profile_pic']
        ],
        'last_message' => [
            'text' => $chat['last_message'],
            'type' => $chat['last_message_type'] ?? 'text',
            'time' => $chat['last_message_time'],
            'id' => $chat['last_message_id'] ? (int)$chat['last_message_id'] : null
        ]
    ];
}, $chats);

echo json_encode(['chats' => $formattedChats]);
