<?php
/**
 * Get or create chat between two users
 * POST: other_user_id
 * CRITICAL: This handles the first message bug fix
 */

header('Content-Type: application/json');
require_once '../auth.php';
require_once '../db.php';

requireAuth();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
$otherUserId = (int)($input['other_user_id'] ?? 0);
$currentUserId = getCurrentUserId();

if ($otherUserId <= 0 || $otherUserId === $currentUserId) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid user ID']);
    exit;
}

$db = getDB();

// Check if other user exists
$stmt = $db->prepare("SELECT id, username, profile_pic_url FROM users WHERE id = ?");
$stmt->execute([$otherUserId]);
$otherUser = $stmt->fetch();

if (!$otherUser) {
    http_response_code(404);
    echo json_encode(['error' => 'User not found']);
    exit;
}

// Try to find existing chat (check both directions)
$stmt = $db->prepare("
    SELECT id, user1_id, user2_id, created_at 
    FROM chats 
    WHERE (user1_id = ? AND user2_id = ?) OR (user1_id = ? AND user2_id = ?)
    LIMIT 1
");
$stmt->execute([$currentUserId, $otherUserId, $otherUserId, $currentUserId]);
$chat = $stmt->fetch();

if ($chat) {
    // Chat exists, return it
    echo json_encode([
        'chat_id' => (int)$chat['id'],
        'created' => false,
        'other_user' => [
            'id' => (int)$otherUser['id'],
            'username' => $otherUser['username'],
            'profile_pic_url' => $otherUser['profile_pic_url']
        ]
    ]);
} else {
    // Create new chat (ensure user1_id < user2_id for consistency)
    $user1Id = min($currentUserId, $otherUserId);
    $user2Id = max($currentUserId, $otherUserId);
    
    try {
        $stmt = $db->prepare("INSERT INTO chats (user1_id, user2_id) VALUES (?, ?)");
        $stmt->execute([$user1Id, $user2Id]);
        $chatId = $db->lastInsertId();
        
        echo json_encode([
            'chat_id' => (int)$chatId,
            'created' => true,
            'other_user' => [
                'id' => (int)$otherUser['id'],
                'username' => $otherUser['username'],
                'profile_pic_url' => $otherUser['profile_pic_url']
            ]
        ]);
    } catch (PDOException $e) {
        // If duplicate (race condition), fetch existing
        $stmt = $db->prepare("
            SELECT id FROM chats 
            WHERE (user1_id = ? AND user2_id = ?) OR (user1_id = ? AND user2_id = ?)
            LIMIT 1
        ");
        $stmt->execute([$currentUserId, $otherUserId, $otherUserId, $currentUserId]);
        $chat = $stmt->fetch();
        
        echo json_encode([
            'chat_id' => (int)$chat['id'],
            'created' => false,
            'other_user' => [
                'id' => (int)$otherUser['id'],
                'username' => $otherUser['username'],
                'profile_pic_url' => $otherUser['profile_pic_url']
            ]
        ]);
    }
}
