<?php
/**
 * User Registration API
 * POST: username, password, profile_pic_url
 */

header('Content-Type: application/json');
require_once '../db.php';
require_once '../config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);

$username = trim($input['username'] ?? '');
$password = $input['password'] ?? '';
$profile_pic_url = trim($input['profile_pic_url'] ?? '');

// Validation
if (empty($username) || empty($password) || empty($profile_pic_url)) {
    http_response_code(400);
    echo json_encode(['error' => 'Username, password, and profile picture are required']);
    exit;
}

if (strlen($username) < 3 || strlen($username) > 50) {
    http_response_code(400);
    echo json_encode(['error' => 'Username must be between 3 and 50 characters']);
    exit;
}

if (strlen($password) < 6) {
    http_response_code(400);
    echo json_encode(['error' => 'Password must be at least 6 characters']);
    exit;
}

// Validate profile picture URL (must be from external media server)
if (!filter_var($profile_pic_url, FILTER_VALIDATE_URL)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid profile picture URL']);
    exit;
}

$parsed_url = parse_url($profile_pic_url);
if (!isset($parsed_url['host']) || $parsed_url['host'] !== MEDIA_SERVER_DOMAIN) {
    http_response_code(400);
    echo json_encode(['error' => 'Profile picture must be from authorized media server']);
    exit;
}

$db = getDB();

// Check if username already exists
$stmt = $db->prepare("SELECT id FROM users WHERE username = ?");
$stmt->execute([$username]);
if ($stmt->fetch()) {
    http_response_code(409);
    echo json_encode(['error' => 'Username already exists']);
    exit;
}

// Hash password
$password_hash = password_hash($password, PASSWORD_BCRYPT);

// Insert user
try {
    $stmt = $db->prepare("INSERT INTO users (username, password_hash, profile_pic_url) VALUES (?, ?, ?)");
    $stmt->execute([$username, $password_hash, $profile_pic_url]);
    
    $userId = $db->lastInsertId();
    
    // Start session
    session_start();
    session_regenerate_id(true);
    $_SESSION['user_id'] = $userId;
    $_SESSION['username'] = $username;
    
    echo json_encode([
        'success' => true,
        'user' => [
            'id' => $userId,
            'username' => $username,
            'profile_pic_url' => $profile_pic_url
        ]
    ]);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Registration failed']);
}
