<?php
/**
 * Send a message
 * POST: chat_id, message_text (optional), media_url (optional), media_type (optional)
 * CRITICAL: This handles the first message bug fix
 */

header('Content-Type: application/json');
require_once '../auth.php';
require_once '../db.php';
require_once '../config.php';

requireAuth();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
$chatId = isset($input['chat_id']) ? (int)$input['chat_id'] : 0;
$otherUserId = isset($input['other_user_id']) ? (int)$input['other_user_id'] : 0;
$messageText = trim($input['message_text'] ?? '');
$mediaUrl = trim($input['media_url'] ?? '');
$mediaType = $input['media_type'] ?? 'text';

$currentUserId = getCurrentUserId();
$db = getDB();

// If chat_id is 0, we need to create/get chat first (first message scenario)
if ($chatId === 0 && $otherUserId > 0) {
    // Get or create chat
    $stmt = $db->prepare("
        SELECT id FROM chats 
        WHERE (user1_id = ? AND user2_id = ?) OR (user1_id = ? AND user2_id = ?)
        LIMIT 1
    ");
    $stmt->execute([$currentUserId, $otherUserId, $otherUserId, $currentUserId]);
    $chat = $stmt->fetch();
    
    if (!$chat) {
        // Create new chat
        $user1Id = min($currentUserId, $otherUserId);
        $user2Id = max($currentUserId, $otherUserId);
        
        try {
            $stmt = $db->prepare("INSERT INTO chats (user1_id, user2_id) VALUES (?, ?)");
            $stmt->execute([$user1Id, $user2Id]);
            $chatId = $db->lastInsertId();
        } catch (PDOException $e) {
            // Race condition - fetch existing
            $stmt = $db->prepare("
                SELECT id FROM chats 
                WHERE (user1_id = ? AND user2_id = ?) OR (user1_id = ? AND user2_id = ?)
                LIMIT 1
            ");
            $stmt->execute([$currentUserId, $otherUserId, $otherUserId, $currentUserId]);
            $chat = $stmt->fetch();
            $chatId = $chat['id'];
        }
    } else {
        $chatId = $chat['id'];
    }
}

if ($chatId <= 0) {
    http_response_code(400);
    echo json_encode(['error' => 'Chat ID or other user ID is required']);
    exit;
}

// Verify user is part of this chat
$stmt = $db->prepare("SELECT id, user1_id, user2_id FROM chats WHERE id = ? AND (user1_id = ? OR user2_id = ?)");
$stmt->execute([$chatId, $currentUserId, $currentUserId]);
$chat = $stmt->fetch();

if (!$chat) {
    http_response_code(403);
    echo json_encode(['error' => 'Access denied']);
    exit;
}

// Validation
if (empty($messageText) && empty($mediaUrl)) {
    http_response_code(400);
    echo json_encode(['error' => 'Message text or media is required']);
    exit;
}

if (!in_array($mediaType, ['text', 'image', 'video'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid media type']);
    exit;
}

// Validate media URL if provided
if (!empty($mediaUrl)) {
    if (!filter_var($mediaUrl, FILTER_VALIDATE_URL)) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid media URL']);
        exit;
    }
    
    $parsed_url = parse_url($mediaUrl);
    if (!isset($parsed_url['host']) || $parsed_url['host'] !== MEDIA_SERVER_DOMAIN) {
        http_response_code(400);
        echo json_encode(['error' => 'Media must be from authorized media server']);
        exit;
    }
}

// Get other user info for response
$otherUserIdInChat = ($chat['user1_id'] == $currentUserId) ? $chat['user2_id'] : $chat['user1_id'];
$stmt = $db->prepare("SELECT id, username, profile_pic_url FROM users WHERE id = ?");
$stmt->execute([$otherUserIdInChat]);
$otherUser = $stmt->fetch();

// Insert message
$stmt = $db->prepare("
    INSERT INTO messages (chat_id, sender_id, message_text, media_url, media_type) 
    VALUES (?, ?, ?, ?, ?)
");
$stmt->execute([
    $chatId,
    $currentUserId,
    $messageText ?: null,
    $mediaUrl ?: null,
    $mediaType
]);

$messageId = $db->lastInsertId();

// Get the full message with sender info
$stmt = $db->prepare("
    SELECT 
        m.id,
        m.sender_id,
        m.message_text,
        m.media_url,
        m.media_type,
        m.created_at,
        u.username as sender_username,
        u.profile_pic_url as sender_profile_pic
    FROM messages m
    INNER JOIN users u ON m.sender_id = u.id
    WHERE m.id = ?
");
$stmt->execute([$messageId]);
$message = $stmt->fetch();

// Update last seen
updateLastSeen($currentUserId);

// Return message with chat info (critical for first message fix)
echo json_encode([
    'success' => true,
    'message' => [
        'id' => (int)$message['id'],
        'sender_id' => (int)$message['sender_id'],
        'sender_username' => $message['sender_username'],
        'sender_profile_pic' => $message['sender_profile_pic'],
        'is_sender' => true,
        'message_text' => $message['message_text'],
        'media_url' => $message['media_url'],
        'media_type' => $message['media_type'],
        'created_at' => $message['created_at']
    ],
    'chat_id' => (int)$chatId,
    'chat_created' => $chatId > 0, // Will be true if chat was just created
    'other_user' => [
        'id' => (int)$otherUser['id'],
        'username' => $otherUser['username'],
        'profile_pic_url' => $otherUser['profile_pic_url']
    ]
]);
