// Authentication JavaScript

// Check authentication on page load
async function checkAuth() {
    try {
        const data = await apiCall('check_auth.php');
        if (data.authenticated) {
            // Redirect to home if already logged in (except on home/chat pages)
            const currentPage = window.location.pathname.split('/').pop();
            if (currentPage === 'index.html' || currentPage === 'register.html' || currentPage === '') {
                window.location.href = 'home.html';
            }
            return data.user;
        } else {
            // Redirect to login if not authenticated (except on login/register pages)
            const currentPage = window.location.pathname.split('/').pop();
            if (currentPage !== 'index.html' && currentPage !== 'register.html' && currentPage !== '') {
                window.location.href = 'index.html';
            }
            return null;
        }
    } catch (error) {
        console.error('Auth check failed:', error);
        return null;
    }
}

// Login handler
async function handleLogin(username, password) {
    try {
        const data = await apiCall('login.php', {
            method: 'POST',
            body: JSON.stringify({ username, password }),
        });
        
        if (data.success) {
            window.location.href = 'home.html';
        }
    } catch (error) {
        showError('loginError', error.message);
    }
}

// Register handler
async function handleRegister(username, password, profilePicUrl) {
    try {
        const data = await apiCall('register.php', {
            method: 'POST',
            body: JSON.stringify({ username, password, profile_pic_url: profilePicUrl }),
        });
        
        if (data.success) {
            window.location.href = 'home.html';
        }
    } catch (error) {
        showError('registerError', error.message);
    }
}

// Logout handler
async function handleLogout() {
    try {
        await apiCall('logout.php', { method: 'POST' });
        window.location.href = 'index.html';
    } catch (error) {
        console.error('Logout failed:', error);
        window.location.href = 'index.html';
    }
}

// Helper to show errors
function showError(elementId, message) {
    const errorEl = document.getElementById(elementId);
    if (errorEl) {
        errorEl.textContent = message;
        errorEl.classList.remove('hidden');
    }
}

// Initialize auth on page load
document.addEventListener('DOMContentLoaded', async () => {
    await checkAuth();
    
    // Login form handler
    const loginForm = document.getElementById('loginFormElement');
    if (loginForm) {
        loginForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            const username = document.getElementById('loginUsername').value;
            const password = document.getElementById('loginPassword').value;
            await handleLogin(username, password);
        });
    }
    
    // Register form handler
    const registerForm = document.getElementById('registerForm');
    if (registerForm) {
        registerForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            const username = document.getElementById('registerUsername').value;
            const password = document.getElementById('registerPassword').value;
            const profilePicUrl = registerForm.dataset.profilePicUrl;
            
            if (!profilePicUrl) {
                showError('registerError', 'Please upload a profile picture');
                return;
            }
            
            await handleRegister(username, password, profilePicUrl);
        });
    }
    
    // Logout button handler
    const logoutBtn = document.getElementById('logoutBtn');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', handleLogout);
    }
});
