// Chat Page JavaScript
// CRITICAL: Handles first message bug fix - chat must appear instantly

let currentUser = null;
let chatId = null;
let otherUserId = null;
let otherUser = null;
let messages = [];
let lastMessageId = 0;
let pollingInterval = null;
let isFirstMessage = false;

// Get URL parameters
function getUrlParams() {
    const params = new URLSearchParams(window.location.search);
    return {
        chatId: params.get('chat_id'),
        otherUserId: params.get('other_user_id')
    };
}

// Load current user
async function loadCurrentUser() {
    try {
        const data = await apiCall('check_auth.php');
        if (data.authenticated) {
            currentUser = data.user;
        }
    } catch (error) {
        console.error('Failed to load user:', error);
        window.location.href = 'index.html';
    }
}

// Load or create chat
async function loadOrCreateChat() {
    const params = getUrlParams();
    
    if (params.chatId) {
        // Existing chat
        chatId = parseInt(params.chatId);
        await loadChatInfo();
    } else if (params.otherUserId) {
        // New chat - get or create
        otherUserId = parseInt(params.otherUserId);
        isFirstMessage = true;
        
        try {
            const data = await apiCall('get_or_create_chat.php', {
                method: 'POST',
                body: JSON.stringify({ other_user_id: otherUserId }),
            });
            
            chatId = data.chat_id;
            otherUser = data.other_user;
            updateChatHeader();
        } catch (error) {
            alert('Failed to load chat: ' + error.message);
            window.location.href = 'home.html';
            return;
        }
    } else {
        window.location.href = 'home.html';
        return;
    }
}

// Load chat info (when chat_id is provided)
async function loadChatInfo() {
    try {
        const data = await apiCall('chats.php');
        const chat = data.chats.find(c => c.chat_id === chatId);
        
        if (chat) {
            otherUser = chat.other_user;
            otherUserId = otherUser.id;
            updateChatHeader();
        } else {
            window.location.href = 'home.html';
        }
    } catch (error) {
        console.error('Failed to load chat info:', error);
    }
}

// Update chat header
function updateChatHeader() {
    const headerPic = document.getElementById('otherUserProfilePic');
    const headerName = document.getElementById('otherUserName');
    
    if (headerPic && otherUser) {
        headerPic.src = otherUser.profile_pic_url || 'data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'40\' height=\'40\'%3E%3Crect fill=\'%23e5e7eb\' width=\'40\' height=\'40\'/%3E%3C/svg%3E';
    }
    
    if (headerName && otherUser) {
        headerName.textContent = otherUser.username;
    }
}

// Load messages
async function loadMessages() {
    if (!chatId) return;
    
    try {
        const url = lastMessageId > 0 
            ? `messages.php?chat_id=${chatId}&after_id=${lastMessageId}`
            : `messages.php?chat_id=${chatId}`;
        
        const data = await apiCall(url);
        const newMessages = data.messages || [];
        
        if (newMessages.length > 0) {
            // Append new messages with animation (from polling)
            newMessages.forEach(msg => {
                messages.push(msg);
                appendMessage(msg); // Each new message animates
            });
            lastMessageId = Math.max(...newMessages.map(m => m.id));
            
            // Mark as read (update last seen)
            if (typeof setLastSeenMessageId === 'function') {
                setLastSeenMessageId(chatId, lastMessageId);
            } else {
                // Fallback: use localStorage directly
                localStorage.setItem(`lastSeen_${chatId}`, String(lastMessageId));
            }
        }
    } catch (error) {
        console.error('Failed to load messages:', error);
    }
}

// Render all messages (without animation - for initial load)
function renderMessages() {
    const container = document.getElementById('chatMessages');
    if (!container) return;
    
    if (messages.length === 0) {
        container.innerHTML = '<div class="text-center text-gray-500 dark:text-gray-400 py-8">No messages yet. Start the conversation!</div>';
        return;
    }
    
    // Render all messages without animation class
    container.innerHTML = messages.map(msg => renderSingleMessage(msg, false)).join('');
}

// Send message
async function sendMessage(text = '', mediaUrl = '', mediaType = 'text') {
    if (!text && !mediaUrl) return;
    
    if (!chatId && otherUserId) {
        // First message scenario - send with other_user_id
        try {
            const response = await apiCall('send_message.php', {
                method: 'POST',
                body: JSON.stringify({
                    chat_id: 0,
                    other_user_id: otherUserId,
                    message_text: text,
                    media_url: mediaUrl,
                    media_type: mediaType,
                }),
            });
            
            // CRITICAL FIX: Handle first message response
            if (response.success) {
                chatId = response.chat_id;
                otherUser = response.other_user;
                updateChatHeader();
                
                // Add message to local state
                messages.push(response.message);
                lastMessageId = response.message.id;
                appendMessage(response.message); // Only new message animates
                
                // Update URL without reload
                window.history.replaceState({}, '', `chat.html?chat_id=${chatId}`);
                
                // CRITICAL: Update chat list in home page (if function exists)
                if (typeof window.addChatToList === 'function') {
                    // This will be called from home.js context
                    window.addChatToList({
                        chat_id: chatId,
                        other_user: otherUser,
                        last_message: {
                            text: text || (mediaType === 'image' ? '📷 Image' : '🎥 Video'),
                            type: mediaType,
                            time: response.message.created_at,
                            id: response.message.id,
                        },
                    });
                } else {
                    // If home.js is not loaded, trigger a custom event
                    window.dispatchEvent(new CustomEvent('chatCreated', {
                        detail: {
                            chat_id: chatId,
                            other_user: otherUser,
                            last_message: {
                                text: text || (mediaType === 'image' ? '📷 Image' : '🎥 Video'),
                                type: mediaType,
                                time: response.message.created_at,
                                id: response.message.id,
                            },
                        },
                    }));
                }
                
                // Start polling
                startPolling();
            }
        } catch (error) {
            alert('Failed to send message: ' + error.message);
        }
    } else if (chatId) {
        // Existing chat
        try {
            const response = await apiCall('send_message.php', {
                method: 'POST',
                body: JSON.stringify({
                    chat_id: chatId,
                    message_text: text,
                    media_url: mediaUrl,
                    media_type: mediaType,
                }),
            });
            
            if (response.success) {
                messages.push(response.message);
                lastMessageId = response.message.id;
                appendMessage(response.message); // Only new message animates
            }
        } catch (error) {
            alert('Failed to send message: ' + error.message);
        }
    }
}

// Start polling for new messages
function startPolling() {
    if (pollingInterval) {
        clearInterval(pollingInterval);
    }
    
    pollingInterval = setInterval(() => {
        if (chatId) {
            loadMessages();
        }
    }, 2500); // Poll every 2.5 seconds
}

// Stop polling
function stopPolling() {
    if (pollingInterval) {
        clearInterval(pollingInterval);
        pollingInterval = null;
    }
}

// Format time
function formatTime(timestamp) {
    const date = new Date(timestamp);
    const now = new Date();
    const hours = date.getHours();
    const minutes = date.getMinutes();
    const ampm = hours >= 12 ? 'PM' : 'AM';
    const displayHours = hours % 12 || 12;
    const displayMinutes = minutes.toString().padStart(2, '0');
    
    return `${displayHours}:${displayMinutes} ${ampm}`;
}

// Escape HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Scroll to bottom - smooth and gentle
function scrollToBottom(smooth = true) {
    const container = document.getElementById('chatMessages');
    if (!container) return;
    
    if (smooth) {
        // Smooth scroll - gentle animation
        container.scrollTo({
            top: container.scrollHeight,
            behavior: 'smooth'
        });
    } else {
        // Instant scroll (for initial load)
        container.scrollTop = container.scrollHeight;
    }
}

// Render single message HTML (without animation class)
function renderSingleMessage(msg, includeAnimation = false) {
    const myId = currentUser ? Number(currentUser.id) : null;
    const isSender = myId !== null
        ? Number(msg.sender_id) === myId
        : Boolean(msg.is_sender);
    const profilePic = msg.sender_profile_pic || 'data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'32\' height=\'32\'%3E%3Crect fill=\'%23e5e7eb\' width=\'32\' height=\'32\'/%3E%3C/svg%3E';
    const time = formatTime(msg.created_at);
    
    let messageContent = '';
    
    if (msg.media_type === 'image' && msg.media_url) {
        const ext = (msg.media_url.match(/\.(jpe?g|png|webp|gif)/i) || [])[1] || 'jpg';
        const filename = `chat-image-${msg.id}.${ext}`;
        const imgHtml = `<img src="${msg.media_url}" alt="Image" class="max-w-full rounded-lg" style="max-height: 300px; display: block;">`;
        messageContent = `<a href="${msg.media_url}" download="${filename}" class="chat-img-wrap block mb-1" title="Click to download">${imgHtml}</a>`;
    } else if (msg.media_type === 'video' && msg.media_url) {
        messageContent = `
            <video controls class="max-w-full rounded-lg mb-1" style="max-height: 300px;">
                <source src="${msg.media_url}" type="video/mp4">
                Your browser does not support the video tag.
            </video>
        `;
    }
    
    if (msg.message_text) {
        messageContent += `<p class="mb-1">${escapeHtml(msg.message_text)}</p>`;
    }
    
    const animationClass = includeAnimation ? 'msg-row' : '';
    const ownClass = isSender ? ' msg-own' : '';
    
    if (isSender) {
        return `
            <div class="msg-row-wrap ${animationClass}${ownClass} w-full flex items-end justify-end space-x-2 mb-1" data-msg-id="${msg.id}" data-is-sender="1">
                <div class="message-bubble message-sender p-3 msg-bubble">
                    ${messageContent}
                    <span class="text-xs opacity-75">${time}</span>
                </div>
                <img src="${profilePic}" alt="${msg.sender_username}" class="w-8 h-8 rounded-full object-cover flex-shrink-0">
            </div>
        `;
    } else {
        return `
            <div class="msg-row-wrap ${animationClass} w-full flex items-end justify-start space-x-2 mb-1" data-msg-id="${msg.id}" data-is-sender="0">
                <img src="${profilePic}" alt="${msg.sender_username}" class="w-8 h-8 rounded-full object-cover flex-shrink-0">
                <div class="message-bubble message-receiver p-3">
                    ${messageContent}
                    <span class="text-xs opacity-75">${time}</span>
                </div>
            </div>
        `;
    }
}

// Append single message with animation (only for new messages)
function appendMessage(msg) {
    const container = document.getElementById('chatMessages');
    if (!container) return;
    
    // Remove "No messages" placeholder if exists
    const placeholder = container.querySelector('.text-center');
    if (placeholder) {
        placeholder.remove();
    }
    
    // Create message element
    const messageHTML = renderSingleMessage(msg, true); // true = include animation
    const tempDiv = document.createElement('div');
    tempDiv.innerHTML = messageHTML.trim();
    const messageEl = tempDiv.firstElementChild;
    
    // Append to container
    container.appendChild(messageEl);
    
    // Smooth scroll to bottom
    setTimeout(() => {
        scrollToBottom(true);
    }, 50);
}

// --- Hold to delete ---
let deletePendingFor = null;
let longPressTimer = null;
const LONG_PRESS_MS = 500;

function showDeletePopover(rowEl) {
    const popover = document.getElementById('deleteMsgPopover');
    if (!popover) return;
    const msgId = rowEl.getAttribute('data-msg-id');
    if (!msgId) return;
    
    deletePendingFor = msgId;
    popover.style.left = '-9999px';
    popover.style.top = '0';
    popover.classList.add('visible');
    
    const rect = rowEl.getBoundingClientRect();
    requestAnimationFrame(() => {
        const w = popover.offsetWidth || 90;
        const h = popover.offsetHeight || 40;
        let left = rect.left + (rect.width / 2) - (w / 2);
        let top = rect.top - h - 8;
        if (top < 10) top = rect.bottom + 8;
        if (left < 10) left = 10;
        if (left + w > window.innerWidth - 10) left = window.innerWidth - w - 10;
        popover.style.left = left + 'px';
        popover.style.top = top + 'px';
    });
}

function hideDeletePopover() {
    const popover = document.getElementById('deleteMsgPopover');
    if (popover) popover.classList.remove('visible');
    deletePendingFor = null;
}

async function deleteMessage(msgId) {
    if (!msgId) return;
    try {
        await apiCall('delete_message.php', {
            method: 'POST',
            body: JSON.stringify({ message_id: msgId }),
        });
        const idx = messages.findIndex(m => m.id === Number(msgId));
        if (idx >= 0) messages.splice(idx, 1);
        const row = document.querySelector(`.msg-row-wrap[data-msg-id="${msgId}"]`);
        if (row) {
            row.style.opacity = '0';
            row.style.transform = 'scale(0.96)';
            row.style.transition = 'opacity 0.2s, transform 0.2s';
            setTimeout(() => {
                row.remove();
                const container = document.getElementById('chatMessages');
                if (container && messages.length === 0) {
                    container.innerHTML = '<div class="text-center text-gray-500 dark:text-gray-400 py-8">No messages yet. Start the conversation!</div>';
                }
            }, 200);
        }
        hideDeletePopover();
    } catch (e) {
        alert('Could not delete: ' + (e.message || 'Error'));
    }
}

// User info modal
async function showUserInfo() {
    if (!otherUser) return;
    const modal = document.getElementById('userInfoModal');
    const pic = document.getElementById('infoUserPic');
    const name = document.getElementById('infoUserName');
    const meta = document.getElementById('infoUserMeta');
    if (!modal || !pic || !name || !meta) return;
    
    pic.src = otherUser.profile_pic_url || 'data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'96\' height=\'96\'%3E%3Crect fill=\'%23e5e7eb\' width=\'96\' height=\'96\'/%3E%3C/svg%3E';
    name.textContent = otherUser.username;
    meta.textContent = 'Loading…';
    modal.classList.add('visible');
    
    try {
        const data = await apiCall(`user_info.php?user_id=${otherUser.id}`);
        const u = data.user;
        meta.textContent = u.last_seen ? `Last seen ${formatLastSeen(u.last_seen)}` : 'Chatting with you';
    } catch {
        meta.textContent = 'Chatting with you';
    }
}

function formatLastSeen(ts) {
    const d = new Date(ts);
    const n = new Date();
    const diff = (n - d) / 60000;
    if (diff < 1) return 'just now';
    if (diff < 60) return `${Math.floor(diff)}m ago`;
    if (diff < 1440) return `${Math.floor(diff / 60)}h ago`;
    if (diff < 10080) return `${Math.floor(diff / 1440)}d ago`;
    return d.toLocaleDateString();
}

// Initialize page
document.addEventListener('DOMContentLoaded', async () => {
    // VIP modal (Audio/Video call) – setup first so it works even if async init fails
    const vipModal = document.getElementById('vipModal');
    const vipModalBtn = document.getElementById('vipModalBtn');
    const chatAudioCallBtn = document.getElementById('chatAudioCallBtn');
    const chatVideoCallBtn = document.getElementById('chatVideoCallBtn');
    const showVipModal = (e) => {
        if (e) { e.preventDefault(); e.stopPropagation(); }
        if (vipModal) { vipModal.style.display = 'flex'; }
    };
    const hideVipModal = () => {
        if (vipModal) { vipModal.style.display = 'none'; }
    };
    if (chatAudioCallBtn) chatAudioCallBtn.addEventListener('click', showVipModal);
    if (chatVideoCallBtn) chatVideoCallBtn.addEventListener('click', showVipModal);
    if (vipModalBtn) vipModalBtn.addEventListener('click', hideVipModal);
    if (vipModal) {
        vipModal.addEventListener('click', (e) => {
            if (e.target === vipModal) hideVipModal();
        });
    }

    await loadCurrentUser();
    await loadOrCreateChat();
    await loadMessages();
    renderMessages(); // Initial load - no animation
    scrollToBottom(false); // Instant scroll for initial load
    
    // Mark all messages as read when opening chat
    if (messages.length > 0 && chatId) {
        const latestId = Math.max(...messages.map(m => m.id));
        localStorage.setItem(`lastSeen_${chatId}`, String(latestId));
    }
    
    startPolling();
    
    // Send button
    const sendBtn = document.getElementById('sendBtn');
    const messageInput = document.getElementById('messageInput');
    
    if (sendBtn && messageInput) {
        const handleSend = () => {
            const text = messageInput.value.trim();
            if (text) {
                sendMessage(text);
                messageInput.value = '';
            }
        };
        
        sendBtn.addEventListener('click', handleSend);
        messageInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                handleSend();
            }
        });
    }
    
    // Image upload
    const attachImageBtn = document.getElementById('attachImageBtn');
    const imageInput = document.getElementById('imageInput');
    
    if (attachImageBtn && imageInput) {
        attachImageBtn.addEventListener('click', () => {
            imageInput.click();
        });
        
        imageInput.addEventListener('change', async (e) => {
            const file = e.target.files[0];
            if (!file) return;
            
            if (!file.type.startsWith('image/')) {
                alert('Please select an image file');
                return;
            }
            
            if (file.size > 20 * 1024 * 1024) {
                alert('Image size must be less than 20 MB');
                return;
            }
            
            try {
                const url = await uploadMedia(file, 'image');
                await sendMessage('', url, 'image');
            } catch (error) {
                alert('Upload failed: ' + error.message);
            }
            
            imageInput.value = '';
        });
    }
    
    // Video upload
    const attachVideoBtn = document.getElementById('attachVideoBtn');
    const videoInput = document.getElementById('videoInput');
    
    if (attachVideoBtn && videoInput) {
        attachVideoBtn.addEventListener('click', () => {
            videoInput.click();
        });
        
        videoInput.addEventListener('change', async (e) => {
            const file = e.target.files[0];
            if (!file) return;
            
            if (file.type !== 'video/mp4') {
                alert('Please select an MP4 video file');
                return;
            }
            
            if (file.size > 500 * 1024 * 1024) {
                alert('Video size must be less than 500 MB');
                return;
            }
            
            try {
                const url = await uploadMedia(file, 'video');
                await sendMessage('', url, 'video');
            } catch (error) {
                alert('Upload failed: ' + error.message);
            }
            
            videoInput.value = '';
        });
    }

    document.getElementById('chatInfoBtn')?.addEventListener('click', showUserInfo);
    document.getElementById('closeUserInfo')?.addEventListener('click', () => {
        document.getElementById('userInfoModal')?.classList.remove('visible');
    });
    document.getElementById('userInfoModal')?.addEventListener('click', (e) => {
        if (e.target.id === 'userInfoModal') e.target.classList.remove('visible');
    });

    // Image click → auto download (delegated)
    const chatMessagesEl = document.getElementById('chatMessages');
    if (chatMessagesEl) {
        chatMessagesEl.addEventListener('click', async (e) => {
            if (deletePendingFor) {
                const row = e.target.closest('.msg-own');
                if (row && row.getAttribute('data-msg-id') === String(deletePendingFor)) {
                    e.preventDefault();
                    e.stopPropagation();
                    hideDeletePopover();
                    return;
                }
            }
            const link = e.target.closest('a.chat-img-wrap');
            if (!link) return;
            e.preventDefault();
            const url = link.getAttribute('href');
            const filename = link.getAttribute('download') || 'chat-image.jpg';
            try {
                const res = await fetch(url);
                const blob = await res.blob();
                const blobUrl = URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = blobUrl;
                a.download = filename;
                a.style.display = 'none';
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                URL.revokeObjectURL(blobUrl);
            } catch (err) {
                window.open(url, '_blank');
            }
        });
    }

    // Hold message → show Delete option
    function clearLongPressTimer() {
        if (longPressTimer) {
            clearTimeout(longPressTimer);
            longPressTimer = null;
        }
    }

    function startHold(row) {
        if (!row || row.getAttribute('data-is-sender') !== '1') return;
        clearLongPressTimer();
        longPressTimer = setTimeout(() => {
            longPressTimer = null;
            showDeletePopover(row);
        }, LONG_PRESS_MS);
    }

    if (chatMessagesEl) {
        chatMessagesEl.addEventListener('mousedown', (e) => {
            const row = e.target.closest('.msg-own');
            if (row) startHold(row);
        });
        chatMessagesEl.addEventListener('mouseup', clearLongPressTimer);
        chatMessagesEl.addEventListener('mouseleave', clearLongPressTimer);

        chatMessagesEl.addEventListener('touchstart', (e) => {
            const row = e.target.closest('.msg-own');
            if (row) startHold(row);
        }, { passive: true });
        chatMessagesEl.addEventListener('touchend', clearLongPressTimer, { passive: true });
        chatMessagesEl.addEventListener('touchmove', clearLongPressTimer, { passive: true });

        chatMessagesEl.addEventListener('contextmenu', (e) => {
            if (e.target.closest('.msg-own')) e.preventDefault();
        });
    }

    document.getElementById('deleteMsgBtn')?.addEventListener('click', () => {
        if (deletePendingFor) deleteMessage(deletePendingFor);
    });

    document.addEventListener('click', (e) => {
        if (!deletePendingFor) return;
        const popover = document.getElementById('deleteMsgPopover');
        if (!popover?.classList.contains('visible')) return;
        if (popover.contains(e.target) || e.target.closest('.msg-own[data-msg-id="' + deletePendingFor + '"]')) return;
        hideDeletePopover();
    });
});

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
    stopPolling();
});
