// Home Page JavaScript
// Handles chat list and find user functionality

let currentUser = null;
let chats = [];
let homePollingInterval = null;
let blockTarget = null;
let blockLongPressTimer = null;
let blockLongPressTriggered = false;
const BLOCK_LONG_PRESS_MS = 500;

// --- Unread message tracking ---
function getLastSeenMessageId(chatId) {
    const key = `lastSeen_${chatId}`;
    const stored = localStorage.getItem(key);
    return stored ? parseInt(stored) : 0;
}

function setLastSeenMessageId(chatId, messageId) {
    const key = `lastSeen_${chatId}`;
    localStorage.setItem(key, String(messageId));
}

function hasUnreadMessages(chat) {
    if (!chat.last_message || !chat.last_message.id) return false;
    const lastSeen = getLastSeenMessageId(chat.chat_id);
    return chat.last_message.id > lastSeen;
}

// Load current user info
async function loadCurrentUser() {
    try {
        const data = await apiCall('check_auth.php');
        if (data.authenticated) {
            currentUser = data.user;
        }
    } catch (error) {
        console.error('Failed to load user:', error);
    }
}

// Load chat list (keeps existing chats on error - fixes "old chats hide" bug)
async function loadChats() {
    try {
        const data = await apiCall('chats.php');
        const list = data.chats || [];
        chats = list;
        renderChatList();
    } catch (error) {
        console.error('Failed to load chats:', error);
        // Don't overwrite chats - keep existing list visible
        if (chats.length > 0) renderChatList();
    }
}

// Render chat list (all chats visible, none hidden)
function renderChatList() {
    const chatList = document.getElementById('chatList');
    if (!chatList) return;

    if (chats.length === 0) {
        chatList.innerHTML = `
            <div class="p-8 text-center text-gray-500 dark:text-gray-400">
                <p>No chats yet. Click + to start a conversation.</p>
            </div>
        `;
        return;
    }

    chatList.innerHTML = chats.map(chat => {
        const lastMessage = chat.last_message;
        const time = lastMessage.time ? formatTime(lastMessage.time) : '';
        const preview = getMessagePreview(lastMessage);
        const hasUnread = hasUnreadMessages(chat);
        const unreadClass = hasUnread ? 'bg-blue-50 dark:bg-blue-900/20 border-l-4 border-blue-500' : '';
        const usernameClass = hasUnread ? 'font-bold text-gray-900 dark:text-white' : 'font-semibold text-gray-800 dark:text-gray-200';
        const previewClass = hasUnread ? 'font-medium text-gray-900 dark:text-gray-200' : 'text-sm text-gray-600 dark:text-gray-400';
        
        return `
            <div class="chat-item ${unreadClass} w-full p-4 transition-all flex items-center gap-2" data-chat-id="${chat.chat_id}" data-other-user-id="${chat.other_user.id}">
                <div class="chat-holder flex items-center flex-1 min-w-0 space-x-3 py-1 -my-1 px-1 -mx-1 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700/50" style="cursor: pointer; -webkit-user-select: none; user-select: none; -webkit-touch-callout: none;" title="Hold for block option">
                    <div class="relative flex-shrink-0">
                        <img src="${chat.other_user.profile_pic_url || 'data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'50\' height=\'50\'%3E%3Crect fill=\'%23e5e7eb\' width=\'50\' height=\'50\'/%3E%3C/svg%3E'}" 
                            alt="${escapeHtml(chat.other_user.username)}" 
                            class="w-12 h-12 rounded-full object-cover ${hasUnread ? 'ring-2 ring-blue-400' : ''}">
                        ${hasUnread ? '<div class="absolute -top-1 -right-1 w-4 h-4 bg-blue-500 rounded-full border-2 border-white dark:border-gray-800 animate-pulse"></div>' : ''}
                    </div>
                    <div class="flex-1 min-w-0">
                        <div class="flex items-center justify-between">
                            <h3 class="${usernameClass} truncate">${escapeHtml(chat.other_user.username)}</h3>
                            <div class="flex items-center gap-2">
                                ${hasUnread ? '<span class="px-2 py-0.5 bg-blue-500 text-white text-xs font-bold rounded-full">NEW</span>' : ''}
                                <span class="text-xs text-gray-500 dark:text-gray-400 ml-2">${time}</span>
                            </div>
                        </div>
                        <p class="${previewClass} truncate">${preview}</p>
                    </div>
                </div>
                <div class="chat-open flex-shrink-0 cursor-pointer p-2 rounded-full hover:bg-gray-100 dark:hover:bg-gray-700 text-gray-500 dark:text-gray-400" title="Open chat">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path></svg>
                </div>
            </div>
        `;
    }).join('');
    
    const blockPopover = document.getElementById('blockPopover');

    function showBlockPopover(chat, ev) {
        blockTarget = { chatId: chat.chat_id, otherUserId: chat.other_user.id };
        blockPopover.classList.add('visible');
        const rect = ev.currentTarget.getBoundingClientRect();
        blockPopover.style.left = Math.max(8, rect.right - 160) + 'px';
        blockPopover.style.top = (rect.bottom + 4) + 'px';
    }

    function hideBlockPopover() {
        blockPopover.classList.remove('visible');
        blockTarget = null;
    }

    chatList.querySelectorAll('.chat-holder').forEach((holder, i) => {
        const chat = chats[i];
        if (!chat) return;
        
        let holderLongPressTriggered = false;
        
        // Long-press to show block option (like message delete)
        holder.addEventListener('mousedown', (e) => {
            e.stopPropagation();
            holderLongPressTriggered = false;
            blockLongPressTimer = setTimeout(() => {
                holderLongPressTriggered = true;
                blockLongPressTriggered = true;
                showBlockPopover(chat, e);
            }, BLOCK_LONG_PRESS_MS);
        });
        
        holder.addEventListener('mouseup', (e) => {
            e.stopPropagation();
            if (blockLongPressTimer) {
                clearTimeout(blockLongPressTimer);
                blockLongPressTimer = null;
            }
            // If long-press didn't trigger, it was a quick click - open chat
            if (!holderLongPressTriggered && !blockLongPressTriggered) {
                if (chat.last_message?.id) setLastSeenMessageId(chat.chat_id, chat.last_message.id);
                window.location.href = `chat.html?chat_id=${chat.chat_id}`;
            }
            blockLongPressTriggered = false;
        });
        
        holder.addEventListener('mouseleave', () => {
            if (blockLongPressTimer) {
                clearTimeout(blockLongPressTimer);
                blockLongPressTimer = null;
            }
            blockLongPressTriggered = false;
        });
        
        // Touch events for mobile
        holder.addEventListener('touchstart', (e) => {
            e.stopPropagation();
            holderLongPressTriggered = false;
            blockLongPressTimer = setTimeout(() => {
                holderLongPressTriggered = true;
                blockLongPressTriggered = true;
                showBlockPopover(chat, e);
            }, BLOCK_LONG_PRESS_MS);
        });
        
        holder.addEventListener('touchend', (e) => {
            e.stopPropagation();
            if (blockLongPressTimer) {
                clearTimeout(blockLongPressTimer);
                blockLongPressTimer = null;
            }
            // If long-press didn't trigger, it was a quick tap - open chat
            if (!holderLongPressTriggered && !blockLongPressTriggered) {
                if (chat.last_message?.id) setLastSeenMessageId(chat.chat_id, chat.last_message.id);
                window.location.href = `chat.html?chat_id=${chat.chat_id}`;
            }
            blockLongPressTriggered = false;
        });
        
        holder.addEventListener('touchmove', () => {
            if (blockLongPressTimer) {
                clearTimeout(blockLongPressTimer);
                blockLongPressTimer = null;
            }
            blockLongPressTriggered = false;
        });
        
        // Prevent context menu on long-press
        holder.addEventListener('contextmenu', (e) => {
            e.preventDefault();
        });
    });

    chatList.querySelectorAll('.chat-open').forEach((btn) => {
        const item = btn.closest('.chat-item');
        const chatId = item?.dataset.chatId;
        const chat = chats.find(c => c.chat_id === Number(chatId));
        if (!chat) return;
        btn.addEventListener('click', (e) => {
            e.stopPropagation();
            if (chat.last_message?.id) setLastSeenMessageId(chatId, chat.last_message.id);
            window.location.href = `chat.html?chat_id=${chatId}`;
        });
    });

    chatList.querySelectorAll('.chat-item').forEach((item, i) => {
        const chat = chats[i];
        if (!chat) return;
        item.addEventListener('click', (e) => {
            // Don't handle if clicking on holder (it has its own handler) or open button
            if (e.target.closest('.chat-holder') || e.target.closest('.chat-open')) return;
            if (chat.last_message?.id) setLastSeenMessageId(item.dataset.chatId, chat.last_message.id);
            window.location.href = `chat.html?chat_id=${chat.chat_id}`;
        });
    });

    if (blockPopover && !blockPopover.dataset.listener) {
        blockPopover.dataset.listener = '1';
        blockPopover.addEventListener('click', (e) => e.stopPropagation());
    }
}

// Get message preview text
function getMessagePreview(lastMessage) {
    if (!lastMessage || !lastMessage.id) {
        return 'No messages yet';
    }
    
    if (lastMessage.type === 'image') {
        return '📷 Image';
    } else if (lastMessage.type === 'video') {
        return '🎥 Video';
    } else {
        return escapeHtml(lastMessage.text || '');
    }
}

// Format time
function formatTime(timestamp) {
    const date = new Date(timestamp);
    const now = new Date();
    const diff = now - date;
    const minutes = Math.floor(diff / 60000);
    const hours = Math.floor(diff / 3600000);
    const days = Math.floor(diff / 86400000);
    
    if (minutes < 1) return 'Just now';
    if (minutes < 60) return `${minutes}m ago`;
    if (hours < 24) return `${hours}h ago`;
    if (days < 7) return `${days}d ago`;
    
    return date.toLocaleDateString();
}

// Escape HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Find user functionality
async function searchUser(username) {
    try {
        const data = await apiCall(`find_user.php?username=${encodeURIComponent(username)}`);
        return data.user;
    } catch (error) {
        return null;
    }
}

// Add new chat to list (CRITICAL: For first message fix)
function addChatToList(chatData) {
    // Check if chat already exists
    const existingIndex = chats.findIndex(c => c.chat_id === chatData.chat_id);
    
    if (existingIndex >= 0) {
        // Update existing chat
        chats[existingIndex] = chatData;
    } else {
        // Add new chat at the beginning
        chats.unshift(chatData);
    }
    
    renderChatList();
}

// Start polling for new messages on home page
function startHomePolling() {
    if (homePollingInterval) {
        clearInterval(homePollingInterval);
    }
    
    homePollingInterval = setInterval(async () => {
        try {
            const data = await apiCall('chats.php');
            const newChats = data.chats || [];
            
            // Check if any chat has new messages
            let hasChanges = false;
            newChats.forEach(newChat => {
                const existing = chats.find(c => c.chat_id === newChat.chat_id);
                if (!existing || (newChat.last_message && newChat.last_message.id && 
                    (!existing.last_message || newChat.last_message.id > existing.last_message.id))) {
                    hasChanges = true;
                }
            });
            
            if (hasChanges || newChats.length !== chats.length) {
                // Check which chats have new unread messages
                const newUnreadChats = newChats.filter(newChat => {
                    const existing = chats.find(c => c.chat_id === newChat.chat_id);
                    if (!existing) return false;
                    if (!newChat.last_message || !newChat.last_message.id) return false;
                    const lastSeen = getLastSeenMessageId(newChat.chat_id);
                    return newChat.last_message.id > lastSeen && 
                           (!existing.last_message || newChat.last_message.id > existing.last_message.id);
                });
                
                chats = newChats;
                renderChatList();
                
                // Pulse animation for new unread messages
                newUnreadChats.forEach(chat => {
                    const item = document.querySelector(`.chat-item[data-chat-id="${chat.chat_id}"]`);
                    if (item) {
                        item.classList.add('unread');
                        setTimeout(() => item.classList.remove('unread'), 500);
                    }
                });
            }
        } catch (error) {
            if (error.message === 'Authentication required') {
                stopHomePolling();
                return;
            }
            console.error('Home polling error:', error);
        }
    }, 3000); // Poll every 3 seconds
}

// Stop polling
function stopHomePolling() {
    if (homePollingInterval) {
        clearInterval(homePollingInterval);
        homePollingInterval = null;
    }
}

// Initialize page
document.addEventListener('DOMContentLoaded', async () => {
    await loadCurrentUser();
    await loadChats();
    startHomePolling(); // Start polling for new messages
    
    // Add user button
    const addUserBtn = document.getElementById('addUserBtn');
    const findUserModal = document.getElementById('findUserModal');
    const closeFindUserModal = document.getElementById('closeFindUserModal');
    const searchUserBtn = document.getElementById('searchUserBtn');
    const searchUsername = document.getElementById('searchUsername');
    const searchResults = document.getElementById('searchResults');
    
    // Open modal
    if (addUserBtn && findUserModal) {
        addUserBtn.addEventListener('click', () => {
            findUserModal.classList.remove('hidden');
            searchUsername.value = '';
            searchResults.innerHTML = '';
        });
    }
    
    // Close modal
    if (closeFindUserModal && findUserModal) {
        closeFindUserModal.addEventListener('click', () => {
            findUserModal.classList.add('hidden');
        });
    }
    
    // Click outside to close
    if (findUserModal) {
        findUserModal.addEventListener('click', (e) => {
            if (e.target === findUserModal) {
                findUserModal.classList.add('hidden');
            }
        });
    }
    
    // Search user
    if (searchUserBtn) {
        searchUserBtn.addEventListener('click', async () => {
            const username = searchUsername.value.trim();
            if (!username) {
                searchResults.innerHTML = '<p class="text-red-500 text-sm">Please enter a username</p>';
                return;
            }
            
            searchUserBtn.disabled = true;
            searchUserBtn.textContent = 'Searching...';
            
            const user = await searchUser(username);
            
            searchUserBtn.disabled = false;
            searchUserBtn.textContent = 'Search';
            
            if (user) {
                searchResults.innerHTML = `
                    <div class="flex items-center space-x-3 p-3 bg-gray-50 rounded-xl border border-gray-100 hover:bg-gray-100/80 transition-all duration-200">
                        <img src="${user.profile_pic_url || 'data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'50\' height=\'50\'%3E%3Crect fill=\'%23e5e7eb\' width=\'50\' height=\'50\'/%3E%3C/svg%3E'}" 
                            alt="${user.username}" 
                            class="w-12 h-12 rounded-full object-cover ring-2 ring-white shadow">
                        <div class="flex-1">
                            <h3 class="font-semibold text-gray-800">${escapeHtml(user.username)}</h3>
                        </div>
                        <button class="message-user-btn flex items-center gap-2 px-4 py-2.5 bg-blue-500 text-white rounded-xl hover:bg-blue-600 hover:scale-105 active:scale-95 transition-all duration-200 font-medium shadow-sm"
                            data-user-id="${user.id}" data-username="${escapeHtml(user.username)}" data-profile-pic="${user.profile_pic_url || ''}">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"></path></svg>
                            Message
                        </button>
                    </div>
                `;
                
                // Message button handler
                const messageBtn = searchResults.querySelector('.message-user-btn');
                if (messageBtn) {
                    messageBtn.addEventListener('click', () => {
                        const userId = messageBtn.dataset.userId;
                        window.location.href = `chat.html?other_user_id=${userId}`;
                    });
                }
            } else {
                searchResults.innerHTML = '<p class="text-red-500 text-sm">User not found</p>';
            }
        });
    }
    
    // Enter key to search
    if (searchUsername) {
        searchUsername.addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && searchUserBtn) {
                searchUserBtn.click();
            }
        });
    }

    // VIP modal (VIP button)
    const vipModal = document.getElementById('vipModal');
    const vipModalBtn = document.getElementById('vipModalBtn');
    document.getElementById('vipBtn')?.addEventListener('click', () => {
        if (vipModal) vipModal.classList.remove('hidden');
    });
    vipModalBtn?.addEventListener('click', () => {
        if (vipModal) vipModal.classList.add('hidden');
    });
    vipModal?.addEventListener('click', (e) => {
        if (e.target === vipModal) vipModal.classList.add('hidden');
    });

    // Block user button (single listener)
    document.getElementById('blockUserBtn')?.addEventListener('click', async (e) => {
        e.stopPropagation();
        if (!blockTarget) return;
        try {
            await apiCall('block_user.php', {
                method: 'POST',
                body: JSON.stringify({ user_id: blockTarget.otherUserId, action: 'block' }),
            });
            document.getElementById('blockPopover')?.classList.remove('visible');
            blockTarget = null;
            await loadChats();
        } catch (err) {
            alert('Could not block: ' + (err.message || 'Error'));
        }
    });

    document.addEventListener('click', () => {
        const pop = document.getElementById('blockPopover');
        if (pop?.classList.contains('visible')) {
            pop.classList.remove('visible');
            blockTarget = null;
        }
    });

    // Account modal handlers
    const accountNavBtn = document.getElementById('accountNavBtn');
    const accountModal = document.getElementById('accountModal');
    const closeAccountModal = document.getElementById('closeAccountModal');
    const accountLogoutBtn = document.getElementById('accountLogoutBtn');
    const homeNavBtn = document.getElementById('homeNavBtn');

    // Update nav active state
    function setNavActive(active) {
        const homeIcon = document.getElementById('homeNavIcon');
        const homeText = document.getElementById('homeNavText');
        const accountIcon = document.getElementById('accountNavIcon');
        const accountText = document.getElementById('accountNavText');
        
        if (active === 'home') {
            homeIcon?.classList.remove('text-gray-500', 'dark:text-gray-400');
            homeIcon?.classList.add('text-blue-500');
            homeText?.classList.remove('text-gray-500', 'dark:text-gray-400');
            homeText?.classList.add('text-blue-500');
            accountIcon?.classList.remove('text-blue-500');
            accountIcon?.classList.add('text-gray-500', 'dark:text-gray-400');
            accountText?.classList.remove('text-blue-500');
            accountText?.classList.add('text-gray-500', 'dark:text-gray-400');
        } else if (active === 'account') {
            accountIcon?.classList.remove('text-gray-500', 'dark:text-gray-400');
            accountIcon?.classList.add('text-blue-500');
            accountText?.classList.remove('text-gray-500', 'dark:text-gray-400');
            accountText?.classList.add('text-blue-500');
            homeIcon?.classList.remove('text-blue-500');
            homeIcon?.classList.add('text-gray-500', 'dark:text-gray-400');
            homeText?.classList.remove('text-blue-500');
            homeText?.classList.add('text-gray-500', 'dark:text-gray-400');
        }
    }

    // Set home as active by default
    setNavActive('home');

    // Open account modal
    if (accountNavBtn && accountModal) {
        accountNavBtn.addEventListener('click', async () => {
            setNavActive('account');
            accountModal.classList.remove('hidden');
            await loadAccountInfo();
        });
    }

    // Close account modal
    if (closeAccountModal && accountModal) {
        closeAccountModal.addEventListener('click', () => {
            accountModal.classList.add('hidden');
            setNavActive('home');
        });
    }

    // Click outside to close account modal
    if (accountModal) {
        accountModal.addEventListener('click', (e) => {
            if (e.target === accountModal) {
                accountModal.classList.add('hidden');
                setNavActive('home');
            }
        });
    }

    // Logout from account modal
    if (accountLogoutBtn) {
        accountLogoutBtn.addEventListener('click', async () => {
            try {
                await apiCall('logout.php', { method: 'POST' });
            } catch (err) {
                console.warn('Logout API error:', err.message);
            }
            window.location.href = 'index.html';
        });
    }

    // Home button - scroll to top and close account modal
    if (homeNavBtn) {
        homeNavBtn.addEventListener('click', () => {
            setNavActive('home');
            if (accountModal && !accountModal.classList.contains('hidden')) {
                accountModal.classList.add('hidden');
            }
            const container = document.getElementById('chatListContainer');
            if (container) {
                container.scrollTo({ top: 0, behavior: 'smooth' });
            }
        });
    }
});

// Load and display account info
async function loadAccountInfo() {
    if (!currentUser) return;
    
    try {
        const data = await apiCall(`user_info.php?user_id=${currentUser.id}`);
        const user = data.user;
        
        document.getElementById('accountProfilePic').src = user.profile_pic_url || 'data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'96\' height=\'96\'%3E%3Crect fill=\'%23e5e7eb\' width=\'96\' height=\'96\'/%3E%3C/svg%3E';
        document.getElementById('accountUsername').textContent = user.username;
        document.getElementById('accountUserIdValue').textContent = user.id;
        
        const createdAt = new Date(user.created_at);
        document.getElementById('accountCreatedAt').textContent = createdAt.toLocaleString('en-US', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
        
        if (user.last_seen) {
            const lastSeen = new Date(user.last_seen);
            const now = new Date();
            const diff = (now - lastSeen) / 60000; // minutes
            
            let lastSeenText = '';
            if (diff < 1) {
                lastSeenText = 'Just now';
            } else if (diff < 60) {
                lastSeenText = `${Math.floor(diff)} minutes ago`;
            } else if (diff < 1440) {
                lastSeenText = `${Math.floor(diff / 60)} hours ago`;
            } else {
                lastSeenText = lastSeen.toLocaleString('en-US', {
                    year: 'numeric',
                    month: 'long',
                    day: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                });
            }
            document.getElementById('accountLastSeen').textContent = lastSeenText;
        } else {
            document.getElementById('accountLastSeen').textContent = 'Never';
        }
    } catch (err) {
        console.error('Failed to load account info:', err);
        // Fallback to basic info
        if (currentUser) {
            document.getElementById('accountProfilePic').src = currentUser.profile_pic_url || 'data:image/svg+xml,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'96\' height=\'96\'%3E%3Crect fill=\'%23e5e7eb\' width=\'96\' height=\'96\'/%3E%3C/svg%3E';
            document.getElementById('accountUsername').textContent = currentUser.username;
            document.getElementById('accountUserIdValue').textContent = currentUser.id;
        }
    }
}

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
    stopHomePolling();
});

// Export functions for chat.js to use
window.addChatToList = addChatToList;
window.setLastSeenMessageId = setLastSeenMessageId;
