// Media Upload Handler
// Handles uploads to external media server
// Note: MEDIA_SERVER_URL is defined in config.js

async function uploadMedia(file, type = 'image') {
    return new Promise((resolve, reject) => {
        const formData = new FormData();
        formData.append('file', file);
        formData.append('type', type);
        
        const xhr = new XMLHttpRequest();
        
        // Progress tracking
        xhr.upload.addEventListener('progress', (e) => {
            if (e.lengthComputable) {
                const percentComplete = (e.loaded / e.total) * 100;
                updateUploadProgress(percentComplete);
            }
        });
        
        xhr.addEventListener('load', () => {
            if (xhr.status === 200) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.success) {
                        hideUploadProgress();
                        resolve(response.url);
                    } else {
                        reject(new Error(response.error || 'Upload failed'));
                    }
                } catch (e) {
                    reject(new Error('Invalid response from server'));
                }
            } else {
                try {
                    const response = JSON.parse(xhr.responseText);
                    reject(new Error(response.error || 'Upload failed'));
                } catch (e) {
                    reject(new Error('Upload failed'));
                }
            }
        });
        
        xhr.addEventListener('error', () => {
            reject(new Error('Network error during upload'));
        });
        
        xhr.open('POST', MEDIA_SERVER_URL);
        xhr.send(formData);
    });
}

function updateUploadProgress(percent) {
    const progressContainer = document.getElementById('uploadProgress');
    const progressBar = document.getElementById('uploadProgressBar');
    
    if (progressContainer && progressBar) {
        progressContainer.classList.remove('hidden');
        progressBar.style.width = percent + '%';
    }
}

function hideUploadProgress() {
    const progressContainer = document.getElementById('uploadProgress');
    if (progressContainer) {
        progressContainer.classList.add('hidden');
    }
    const progressBar = document.getElementById('uploadProgressBar');
    if (progressBar) {
        progressBar.style.width = '0%';
    }
}

// Profile picture upload handler (for registration)
function initProfilePicUpload() {
    const profilePicInput = document.getElementById('profilePicInput');
    const selectProfilePicBtn = document.getElementById('selectProfilePicBtn');
    const profilePicPreview = document.getElementById('profilePicPreview');
    const registerForm = document.getElementById('registerForm');
    const registerBtn = document.getElementById('registerBtn');
    
    if (!selectProfilePicBtn) {
        // Not on registration page, skip
        return;
    }
    
    if (!profilePicInput) {
        console.error('Profile picture input not found');
        return;
    }
    
    // The label's 'for' attribute automatically triggers the file input when clicked
    // No additional click handler needed
    
    // Add change event to file input
    profilePicInput.addEventListener('change', async function(e) {
        const file = e.target.files[0];
        if (!file) return;
        
        // Validate file type
        const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/webp'];
        if (!allowedTypes.includes(file.type)) {
            alert('Please select a valid image file (JPG, PNG, or WebP)');
            profilePicInput.value = ''; // Clear the input
            return;
        }
        
        // Validate file size (10 MB)
        if (file.size > 10 * 1024 * 1024) {
            alert('File size must be less than 10 MB');
            profilePicInput.value = ''; // Clear the input
            return;
        }
        
        // Show preview
        if (profilePicPreview) {
            const reader = new FileReader();
            reader.onload = function(e) {
                profilePicPreview.src = e.target.result;
            };
            reader.readAsDataURL(file);
        }
        
        // Upload to media server
        try {
            if (registerBtn) {
                registerBtn.disabled = true;
                registerBtn.textContent = 'Uploading...';
            }
            
            const url = await uploadMedia(file, 'profile_pic');
            
            // Store URL in form dataset
            if (registerForm) {
                registerForm.dataset.profilePicUrl = url;
            }
            
            if (registerBtn) {
                registerBtn.disabled = false;
                registerBtn.textContent = 'Create Account';
            }
        } catch (error) {
            alert('Upload failed: ' + error.message);
            if (registerBtn) {
                registerBtn.disabled = false;
                registerBtn.textContent = 'Create Account';
            }
            profilePicInput.value = ''; // Clear the input on error
        }
    });
}

// Initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initProfilePicUpload);
} else {
    // DOM is already loaded
    initProfilePicUpload();
}
