<?php
/**
 * External Media Server Upload Handler
 * Handles profile pictures, chat images, and videos
 * 
 * IMPORTANT: This file should be deployed on your external media server
 * Update the upload directory path and domain whitelist as needed
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Configuration
$UPLOAD_DIR = __DIR__ . '/uploads/';
$MAX_PROFILE_PIC_SIZE = 10 * 1024 * 1024; // 10 MB
$MAX_IMAGE_SIZE = 20 * 1024 * 1024; // 20 MB
$MAX_VIDEO_SIZE = 500 * 1024 * 1024; // 500 MB

$ALLOWED_PROFILE_PIC_TYPES = ['image/jpeg', 'image/jpg', 'image/png', 'image/webp'];
$ALLOWED_IMAGE_TYPES = ['image/jpeg', 'image/jpg', 'image/png', 'image/webp', 'image/gif'];
$ALLOWED_VIDEO_TYPES = ['video/mp4'];

// Create upload directory if it doesn't exist
if (!file_exists($UPLOAD_DIR)) {
    mkdir($UPLOAD_DIR, 0755, true);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
    http_response_code(400);
    echo json_encode(['error' => 'No file uploaded or upload error']);
    exit;
}

$file = $_FILES['file'];
$uploadType = $_POST['type'] ?? 'image'; // 'profile_pic', 'image', 'video'
$fileSize = $file['size'];
$fileType = $file['type'];
$tmpName = $file['tmp_name'];

// Validate upload type
$allowedTypes = [];
$maxSize = 0;

switch ($uploadType) {
    case 'profile_pic':
        $allowedTypes = $ALLOWED_PROFILE_PIC_TYPES;
        $maxSize = $MAX_PROFILE_PIC_SIZE;
        break;
    case 'image':
        $allowedTypes = $ALLOWED_IMAGE_TYPES;
        $maxSize = $MAX_IMAGE_SIZE;
        break;
    case 'video':
        $allowedTypes = $ALLOWED_VIDEO_TYPES;
        $maxSize = $MAX_VIDEO_SIZE;
        break;
    default:
        http_response_code(400);
        echo json_encode(['error' => 'Invalid upload type']);
        exit;
}

// Validate file type
if (!in_array($fileType, $allowedTypes)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid file type']);
    exit;
}

// Validate file size
if ($fileSize > $maxSize) {
    http_response_code(400);
    $maxSizeMB = round($maxSize / 1024 / 1024, 1);
    echo json_encode(['error' => "File size exceeds maximum of {$maxSizeMB} MB"]);
    exit;
}

// Generate unique filename
$extension = pathinfo($file['name'], PATHINFO_EXTENSION);
$filename = uniqid() . '_' . time() . '.' . $extension;
$filepath = $UPLOAD_DIR . $filename;

// Move uploaded file
if (!move_uploaded_file($tmpName, $filepath)) {
    http_response_code(500);
    echo json_encode(['error' => 'Failed to save file']);
    exit;
}

// Generate public URL — use HTTP (see config USE_HTTP)
$protocol = 'http';
if (is_file(__DIR__ . '/../config.php')) {
    require_once __DIR__ . '/../config.php';
    if (defined('APP_PROTOCOL')) {
        $protocol = APP_PROTOCOL;
    }
}
$host = $_SERVER['HTTP_HOST'];
$publicUrl = "{$protocol}://{$host}/chat_app_pro/media_server/uploads/{$filename}";

echo json_encode([
    'success' => true,
    'url' => $publicUrl,
    'filename' => $filename,
    'size' => $fileSize,
    'type' => $fileType
]);
